<?php

namespace Woovi\Pix\Model\Quote\Address\Total;

use Woovi\Pix\Api\WooviManagementInterface;

class Discount extends \Magento\Quote\Model\Quote\Address\Total\AbstractTotal
{
    /**
     * @var \Woovi\Pix\Api\WooviManagementInterface
     */
    private $wooviManagement;

    /**
     * @var \Magento\Customer\Model\SessionFactory
     */
    private $customerSessionFactory;

    /**
     * @var \Woovi\Pix\Helper\Data
     */
    private $helperData;

    /**
     * Constructor
     *
     * @param \Woovi\Pix\Api\WooviManagementInterface $wooviManagement
     * @param \Magento\Customer\Model\SessionFactory      $customerSessionFactory
     * @param \Woovi\Pix\Helper\Data                    $helperData
     */
    public function __construct(
        \Woovi\Pix\Api\WooviManagementInterface $wooviManagement,
        \Magento\Customer\Model\SessionFactory $customerSessionFactory,
        \Woovi\Pix\Helper\Data $helperData
    ) {
        $this->setCode(WooviManagementInterface::WOOVI_DISCOUNT_CODE);
        $this->wooviManagement = $wooviManagement;
        $this->customerSessionFactory = $customerSessionFactory;
        $this->helperData = $helperData;
    }

    /**
     * @inheritdoc
     */
    public function collect(
        \Magento\Quote\Model\Quote $quote,
        \Magento\Quote\Api\Data\ShippingAssignmentInterface $shippingAssignment,
        \Magento\Quote\Model\Quote\Address\Total $total
    ) {
        if (!$this->helperData->getWooviEnabled()) {
            return false;
        }

        $items = $shippingAssignment->getItems();
        if (!count($items)) {
            return $this;
        }

        parent::collect($quote, $shippingAssignment, $total);

        $paymentMethod = $quote->getPayment()->getMethod();
        if (!$this->validatePaymentMethod($paymentMethod)) {
            $this->updateTotal($quote, $total, 0);
            return $this;
        }

        $discountAmount = $this->getDiscountFromCustomer(
            $quote,
            $total->getDiscountAmount()
        );
        if ($discountAmount <= 0) {
            $this->updateTotal($quote, $total, 0);
            return $this;
        }

        $this->updateTotal($quote, $total, $discountAmount);

        return $this;
    }

    /**
     * Update total
     *
     * @param $quote
     * @param $total
     * @param $amount
     *
     * @return void
     */
    protected function updateTotal($quote, $total, $amount)
    {
        $quote->setWooviDiscount($amount)->setBaseWooviDiscount($amount);

        $total->setWooviDiscount($amount)->setBaseWooviDiscount($amount);

        $total->addTotalAmount($this->getCode(), -$amount);
        $total->addBaseTotalAmount($this->getCode(), -$amount);
    }

    /**
     * @inheritdoc
     */
    public function fetch(
        \Magento\Quote\Model\Quote $quote,
        \Magento\Quote\Model\Quote\Address\Total $total
    ) {
        $result = null;
        $amount = $total->getWooviDiscount();
        if ($amount != 0) {
            $result = [
                'code' => $this->getCode(),
                'title' => __('Giftback Discount'),
                'value' => -$amount,
            ];
        }
        return $result;
    }

    /**
     * Validate payment method is woovi or not
     *
     * @param string $paymentMethod
     *
     * @return bool
     */
    protected function validatePaymentMethod($paymentMethod)
    {
        if (empty($paymentMethod)) {
            return false;
        }

        return strcasecmp($paymentMethod, \Woovi\Pix\Model\Pix\Pix::CODE) == 0;
    }

    /**
     * Get discount from customer balance of OpendPix API
     *
     * @param $quote
     * @param float $totalDiscountAmount
     *
     * @return float
     */
    protected function getDiscountFromCustomer($quote, $totalDiscountAmount)
    {
        $customerSession = $this->customerSessionFactory->create();
        if (!$customerSession->isLoggedIn()) {
            $this->helperData->log(
                'Pix::collect - customer not logged',
                WooviManagementInterface::WOOVI_LOG_NAME
            );
            return 0;
        }

        $customerTaxVat = $quote->getCustomerTaxvat();
        if (!$customerTaxVat) {
            $this->helperData->log(
                'Pix::collect - customer does not have taxID',
                WooviManagementInterface::WOOVI_LOG_NAME
            );
            return 0;
        }

        $this->helperData->log(
            'Pix::collect - customer taxvat ' . $customerTaxVat,
            WooviManagementInterface::WOOVI_LOG_NAME
        );

        try {
            $customerGiftBackBalance = $this->wooviManagement->getCustomerBalanceByTaxId(
                $customerTaxVat
            );
        } catch (\Exception $exception) {
            $customerGiftBackBalance = 0;
        }

        if ($customerGiftBackBalance <= 0) {
            $this->helperData->log(
                'Pix::collect - customer does not have balance. Balance: ' .
                    $customerGiftBackBalance,
                WooviManagementInterface::WOOVI_LOG_NAME
            );
            return 0;
        }

        $this->helperData->log(
            'Pix::collect - giftback balance ' . $customerGiftBackBalance,
            WooviManagementInterface::WOOVI_LOG_NAME
        );

        return $this->wooviManagement->calculateAndConvertBalance(
            $customerGiftBackBalance,
            $quote,
            $totalDiscountAmount
        );
    }
}
